function [ PVHS ] = Function_DropPressureVolume_V03( Rm, nb, zsign, display )
%Function_Pendant_Drop_PressureVolume_V01
% This functions calculates the pressure - volume relationship for a
% pendant drop:
% INPUTS:
% - Rm      radius 
% - nb      number of point in the pressure volume table
% - zsign   % 1 for pendant drop, -1 for pendant bubble
% - display turns the display on or off

global rho g gamma ;

K             = Rm * (rho * g / gamma)^0.5 ; % shape factor
deltapmax     = 2                          ; % Maximum drop size
ns            = 1000                       ; % number of steps in arc length

deltapSpace   = linspace(0,deltapmax,nb)   ; % Indexes list
height        = Rm * deltapSpace           ; % list of heights
pressures     = zeros(nb,1)                ; % list of pressures
volumes       = zeros(nb,1)                ; % list of volumes
surfaces      = zeros(nb,1)                ; % list of surface areas

% the next two parameters can be adjusted to integral numbers larger than 1
% to make sure that sensible initial guesses are always obtained
ndeltapoints = 1 ;  % number of steps used to reach target delta

% Do a for loop to cover all size drops
for i = 1:nb
    
    % Get the drop size
    deltaptarget   = deltapSpace(i)            ; % Target drop size at the tip of the apex
    deldelta       = deltaptarget/ndeltapoints ; % This is the step size
    
    % parmpsol(1) is the overall countour length, normalized by Rm
    % parmpsol(2) is the normalized pressure
    % our initial guess, used only for the lowest value of delta, is a
    % spherical cap
    deltaiter    = 1                   ;
    deltap       = deldelta            ;
    s            = linspace(1/ns,1,ns) ; % this sets the original mesh of points equally spaced along the undeformed arc length
    
    % in the following initial guess we assume that the membrane is deformed to a constant
    % radius of curvature, which is determined by deltap
    amax          = 2*atan(deltap)  ; % this is the angle at s=1 (R=Rm)
    ap            = s*amax          ; 
    rp            = s.*sinc(ap/pi)  ;
    zp            = tan(ap/2)       ;
    solp(1,:)     = ap              ; % these are the angles along the whole membrane profile for the assumed cirular cross section
    solp(2,:)     = rp              ; % these are the r values
    solp(3,:)     = zp              ; % these are the z values
    parmpsol(1)   = 1/sinc(amax/pi) ; % initial guess for lp (normalized arc length of membrane)
    parmpsol(2)   = 4*deltap/(1+deltap^2);  % inital guess for normalized pressure
    
    % in the following loop we solve the pendant drop problem for a constant
    % interfacial tension
    while deltaiter<=ndeltapoints;
        parmpguess      = parmpsol;
        solpguess       = solp;
        [parmpsol,solp] = Function_Pendantliquid_v03(ns,deltap,K,zsign,parmpguess,solpguess);         
        deltaiter       = deltaiter+1;
        deltap          = deltap+deldelta;
    end

    lp   = parmpsol(1)    ; % this is the total arc length, dived by Rm
    p0p  = parmpsol(2)    ; % this is the normalized pressure at the apex

    % the solution to the undeformed case is alrady known:
    rp(:)=solp(2,:);
    zp(:)=solp(3,:);
    
    % Make a figure
    if (display == true)
        figure(10)                 ;
        plot(rp,zp-zp(ns),'--b')   ;
        hold on
        plot(-rp,zp-zp(ns),'--b')  ;
    end

    % Now calculate some of the characteristic gemoetric parameters from the solution
    Vp   = Rm^3 * trapz(zp,pi.*rp.^2)    ; % this is the absolute volume of the undeformed drop
    Ap   = Rm^2 * lp*(trapz(s,2*pi.*rp)) ; % this is the absolute area of the deformed drop
    Pp   = p0p * gamma / Rm              ; % absolute pressure

    % Create the pressure volume table
    pressures(i) = Pp  ; % return the pressure
    volumes(i)   = Vp  ; % return the volume
    surfaces(i)  = Ap  ; % return the surface area
end

if (display == true)
    figure(11)                      ; 
    plot(deltapSpace,volumes,'b')   ;

    figure(12)                      ;
    plot(deltapSpace,pressures,'r') ;
end

% Find the interpolation
PV   = interp1(volumes, pressures, 'spline', 'pp')   ; % Calculate the PV polynomial
HV   = interp1(volumes, height, 'spline', 'pp')      ; % Calculate the PH polynomial 
SV   = interp1(volumes, surfaces, 'spline', 'pp')    ; % Calculate the SH polynomial 

%PV   = griddedInterpolant(volumes, pressures) ; % Calculate the PV polynomial
%HV   = griddedInterpolant(volumes, height)    ; % Calculate the PH polynomial 
%SV   = griddedInterpolant(volumes, surfaces)  ; % Calculate the SH polynomial 

% Make the structure containing the list of pressures, volumes, heights,
% and the interpolated polynomials to quickly calculate intermediate
% values
PVHS           = struct();
PVHS.pressures = pressures ;
PVHS.volumes   = volumes   ;
PVHS.heights   = height    ;
PVHS.surfaces  = surfaces  ;
PVHS.PV        = PV        ;
PVHS.HV        = HV        ;
PVHS.SV        = SV        ;

end

